/******************************************************************************
 * sl_server -- a server that you connect to and it spits out the signal      *
 *              level from /proc/wavelan/<eth#>/mmc every 1/4 of a second to  *
 *              all network clients connected to it, the client may do what   *
 *              ever it wants with the number                                 *
 *                                                                            *
 * takes the eth# as argv1 eg. "./sl_server 1" if the wavelan card is eth1    *
 * I shall sit on port 12000 by default                                       *
 *                                                                            *
 * Author: Steven Hanley                                                      *
 * Date: 14/04/99                                                             *
 * Last Modified: See ChangeLog                                               *
 ******************************************************************************/

/* this is the sl_server modified by Jean Tourrilhes to use wavelan extensions
   I plan to extend this when I have time, for now releaseing this should
   allow people not using the tridge hacked wavelan driver to use the sl_programs
   (and thus any driver that supports the WE in the kernel)
*/

/* after testing I dont see any reason to change this, even though it is waking 
 * up and checking things 4 times a second even on a 486 dx2/66 the cpu usage
 * is 0 when idle and low even with a fair few clients connected.
 */

/* okay this is a concurrent server but I wont bother forking, shall just 
 * keep all conections in a linked list and loop through them when sending 
 * out the number
 */

#include "sl_shared.h"

#include <ctype.h>
/* Wireless extensions */
/* including this file severelty messed up the uses of sockets between kernel and
   user space definitions, fix when I have time 
#include <linux/wireless.h>
*/

/* the struct's were taken from the kernel source (wireless.h) and modified to 
   be more normal c */
/*
 *	Quality of the link
 */
struct	iw_quality
{
	unsigned char		qual;		/* link quality (SNR or better...) */
	unsigned char		level;		/* signal level */
	unsigned char		noise;		/* noise level */
	unsigned char		updated;	/* Flags to know if updated */
};

/*
 *	Packet discarded in the wireless adapter due to
 *	"wireless" specific problems...
 */
struct	iw_discarded
{
	unsigned int nwid;		/* Wrong nwid */
	unsigned int code;		/* Unable to code/decode */
	unsigned int misc;		/* Others cases */
};

struct	iw_statistics
{
	unsigned char		status;		/* Status
					 * - device dependent for now */

	struct iw_quality	qual;		/* Quality of the link
						 * (instant/mean/max) */
	struct iw_discarded	discard;	/* Packet discarded counts */
};


typedef struct iw_statistics    iwstats;

#define LINESTOSL 10 /* the number of lines in the mmc file to the line with 
                        signal_lvl including that line */
#define FILENAME "/proc/wavelan/eth%d/mmc"
/*#define FILENAME "fake_mmc"*/

#define NO_NEW_MESSAGE_TIMEOUT 4 /* this stops the server from sending 0 0 
                                    once it sees no new message until after
                                    a second has passed */

#define SC struct connection

static int no_new_message_count, last_sig_level, last_sil_level;

/* linked list of conections with a file handle we cant printf to */
SC {
   FILE *sock;
   SC *next;
};

void read_levels (const char *mmcfile, int *rvals) {
   FILE *fh;
   int sig_level, sil_level, i;
   char *c, *tmp, line[SLEN];

   fh = fopen (mmcfile, "r");
   if (! fh) {
      fprintf (stderr, "couldn't open file %s, error: %s\n", 
                       mmcfile, strerror(errno));
      exit (1);
   }

   for (i = 0; i < LINESTOSL; i++) {
      fgets (line, SLEN, fh);
   }
   fclose (fh);

   if (strstr (line, "no new msg")) {
   /* there is no new message so signal level is invalid anyway, send out 0 */
      if (no_new_message_count > NO_NEW_MESSAGE_TIMEOUT) {
      /* make the server wait 1 second after no new message first appears
         before reporting it to clients */
         rvals[0] = 0;
         rvals[1] = 0;
      } else {
         no_new_message_count++;
         rvals[0] =  last_sig_level;
         rvals[1] =  last_sil_level;
      }
   } else {
   /* read the signal and silence levels and set them in rvals */
      c = strchr (line, ':');
      c += 2;
      c = strtok (c, " ");
      sig_level = (int) strtol (c, &tmp, 0);
      invalid_num(*tmp);
   
      c = strtok (NULL, ":");
      c = strchr (c, ':');
      c += 2;
      c = strtok (NULL, " ");
      sil_level = (int) strtol (c, &tmp, 0);
      invalid_num(*tmp);
   
      rvals[0] = sig_level;
      rvals[1] = sil_level;

      last_sil_level = sil_level;
      last_sig_level = sig_level;
      no_new_message_count = 0;
   }
}

/*------------------------------------------------------------------*/
/*
 * Read /proc/net/wireless to get the latest statistics
 */
static int
iw_getstats(const char *        ifname,
            iwstats *           stats)
{
  FILE *        f=fopen("/proc/net/wireless","r");
  char          buf[256];
  char *        bp;
  char *        pt;
  int           t;
  if(f==NULL)
    return -1;
  /* Loop on all devices */
  while(fgets(buf,255,f))
    {
      bp=buf;
      while(*bp&&isspace(*bp))
        bp++;
      /* Is it the good device ? */
      if(strncmp(bp,ifname,strlen(ifname))==0 && bp[strlen(ifname)]==':')
        {
          /* Skip ethX: */
          bp=strchr(bp,':');
          bp++;
          /* -- status -- */
          bp = strtok(bp, " ");
          sscanf(bp, "%X", &t);
          stats->status = (unsigned short) t;
          /* -- link quality -- */
          bp = strtok(NULL, " ");
          pt = strchr(bp,'.');
          if(pt != NULL)
            stats->qual.updated |= 1;
          sscanf(bp, "%d", &t);
          stats->qual.qual = (unsigned char) t;
          /* -- signal level -- */
          bp = strtok(NULL, " ");
          pt = strchr(bp,'.');
          if(pt != NULL)
            stats->qual.updated |= 2;
          sscanf(bp, "%d", &t);
          stats->qual.level = (unsigned char) t;
          /* -- noise level -- */
          bp = strtok(NULL, " ");
          pt = strchr(bp,'.');
          if(pt != NULL)
            stats->qual.updated += 4;
          sscanf(bp, "%d", &t);
          stats->qual.noise = (unsigned char) t;
          /* -- discarded packets -- */
          bp = strtok(NULL, " ");
          sscanf(bp, "%d", &stats->discard.nwid);
          bp = strtok(NULL, " ");
          sscanf(bp, "%d", &stats->discard.code);
          bp = strtok(NULL, " ");
          sscanf(bp, "%d", &stats->discard.misc);
          fclose(f);
          return 0;
        }
    }
  fclose(f);
  return -1;
}

/*------------------------------------------------------------------*/
/*
 * Process statistics
 */
int
read_wireless_levels(const char *       ifname,
                     int *              rvals)
{
  iwstats       stats;
  int           rc;

  memset((char *) &stats, 0, sizeof(iwstats));
  rc = iw_getstats(ifname, &stats);

  /* Hum... I believe that not all hardware have the ability
   * to set the updated flag, starting with the old Wavelan Pcmcia.
   */
  if((stats.qual.updated == 0) &&
     (last_sil_level == stats.qual.noise) &&
     (last_sig_level = stats.qual.level))
    {
      /* there is no new message so signal level is invalid anyway, send out 0 */
      if (no_new_message_count > NO_NEW_MESSAGE_TIMEOUT)
        {
          /* make the server wait 1 second after no new message first appears
             before reporting it to clients */
          rvals[0] = 0;
          rvals[1] = 0;
        }
      else
        {
          no_new_message_count++;
          rvals[0] =  last_sig_level;
          rvals[1] =  last_sil_level;
        }
    }
  else
    {
      /* Copy signal and silence levels in rvals */
      rvals[0] = stats.qual.level;
      rvals[1] = stats.qual.noise;

      last_sil_level = stats.qual.noise;
      last_sig_level = stats.qual.level;
      no_new_message_count = 0;
    }
  return rc;
}

void write_to_clients (int *levels, SC **conn) {
   int pret;

   while (*conn) {
      fprintf ((*conn)->sock, "%d %d\n", levels[0], levels[1]);
      pret = fflush ((*conn)->sock);
      if (pret < 0) {
         /* client is dead (closed connection or dead) */
         SC *tmpcon = *conn;
         fclose ((*conn)->sock);
         *conn = (*conn)->next;
         free (tmpcon);
      } else {
         conn = &(*conn)->next;
      }
   }
}

int main (int argc, char **argv) {

   int devicenum, flags, listensock, clientsock, reuse_addr = 1;
   int  pw = 0;
   int levels[2];
   char *tmp, mmcfile[SLEN], ifname[16];
   struct sockaddr_in servaddr;
   SC *conn = NULL;

   if ((argc <= 1) || (argc > 2)) {
      fprintf (stderr, "incorrect number of arguments use:\n"
                       "\"%s number\" where number is 1 if wavelan is eth1\n", 
                       argv[0]);
      return 1;
   }

   devicenum = (int) strtol (argv[1], &tmp, 0);
   invalid_num(*tmp);

   snprintf (mmcfile, SLEN, FILENAME, devicenum);
   snprintf (ifname, 16, "eth%d", devicenum);
   pw = read_wireless_levels (ifname, levels);

   /* ignore SIGPIPE that happens when the clients disconnect */
   signal (SIGPIPE, SIG_IGN);

   if ( (listensock = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
      fprintf (stderr, "couldn't open socket, error: %s\n", 
                       strerror(errno));
      return 1;
   }

   /* set reuseaddr so the port wont sit in time_wait after being killed or 
      anything, this is the correct way to do this */
   setsockopt (listensock, SOL_SOCKET, SO_REUSEADDR, (char *) &reuse_addr, 
               sizeof(reuse_addr));

   /* make the socket nonblocking */
   flags = fcntl (listensock, F_GETFL, 0); 
   fcntl (listensock, F_SETFL, flags | O_NONBLOCK);

   memset (&servaddr, 0, sizeof(servaddr));
   servaddr.sin_family = AF_INET;
   servaddr.sin_addr.s_addr = htonl(INADDR_ANY);
   servaddr.sin_port = htons(PORT);

   if ( bind (listensock, (SA *) &servaddr, sizeof(servaddr)) ) {
      fprintf (stderr, "couldn't bind socket, error: %s\n", 
                       strerror(errno));
      return 1;
   }

   if ( listen (listensock, SOMAXCONN) ) {
      fprintf (stderr, "couldn't listen on socket, error: %s\n", 
                       strerror(errno));
      return 1;
   }

   for (;;) {
      if ((clientsock = accept (listensock, (SA *) NULL, NULL)) >= 0) {
         SC *tmpcon;
         tmpcon = (SC *) malloc (sizeof(SC));
         tmpcon->next = conn;
         conn = tmpcon;
         conn->sock = fdopen (clientsock, "w");
      } else {
         if (errno != EWOULDBLOCK) {
            fprintf (stderr, "error on accept: %s\n", strerror(errno));
            return 0;
         }
      }
      if (conn) {
         if(pw >= 0) {
            read_wireless_levels (ifname, levels);
         } else {
            read_levels (mmcfile, levels);
         }
         write_to_clients(levels, &conn);
      }
      /* sleep for a quarter of a second */
      usleep (250000);
   }

   return 0;
}
